import bpy
import mathutils

# Define the operator classes first

# Create operators for the functions
class RecalculateNormalsOutsideOperator(bpy.types.Operator):
    bl_idname = "mesh.recalculate_normals_outside"
    bl_label = "Recalculate Normals (Outside)"
    
    def execute(self, context):
        recalculate_normals_outside(self, context)
        return {'FINISHED'}

class EnableAutoSmoothOperator(bpy.types.Operator):
    bl_idname = "mesh.enable_auto_smooth"
    bl_label = "Enable Auto Smooth"
    
    def execute(self, context):
        enable_auto_smooth(self, context)
        return {'FINISHED'}

class DuplicateJoinMergeByDistanceOperator(bpy.types.Operator):
    bl_idname = "mesh.duplicate_join_merge_by_distance"
    bl_label = "Duplicate, Join, Merge by Distance"
    
    def execute(self, context):
        duplicate_join_merge_by_distance(self, context)
        return {'FINISHED'}

class PropagateShapeKeysOperator(bpy.types.Operator):
    bl_idname = "mesh.propagate_shape_keys"
    bl_label = "Propagate Shape Keys"
    
    def execute(self, context):
        propagate_shape_keys(self, context)
        return {'FINISHED'}

class BlendFromShapeKeysOperator(bpy.types.Operator):
    bl_idname = "mesh.blend_from_shape_keys"
    bl_label = "Blend From Shape Keys"
    
    def execute(self, context):
        blend_from_shape_keys(self, context)
        return {'FINISHED'}

class SetNormalsFromFacesOperator(bpy.types.Operator):
    bl_idname = "mesh.set_normals_from_faces"
    bl_label = "Set Normals from Faces"
    
    def execute(self, context):
        set_normals_from_faces(self, context)
        return {'FINISHED'}

class TransferWeightsOperator(bpy.types.Operator):
    bl_idname = "mesh.transfer_weights"
    bl_label = "Transfer Weights"

    def execute(self, context):
        transfer_weights(self, context)
        return {'FINISHED'}

class HeatSeekRiggingNoVertexOperator(bpy.types.Operator):
    bl_idname = "mesh.heat_seek_rigging_no_vertex"
    bl_label = "Heat Seek Rigging (No Vertex)"

    def execute(self, context):
        source_object = bpy.context.active_object
        destination_objects = GetSelectedDestinationObjects()
        
        if not source_object:
            self.report({'ERROR'}, "No active object selected as source.")
            return {'CANCELLED'}
        
        # Automatically add objects without vertex groups to the destination list
        heat_seek_rigging_no_vertex(source_object, destination_objects)

        return {'FINISHED'}

# Define the functions from your original script

def recalculate_normals_outside(self, context):
    for obj in bpy.context.selected_objects:
        if obj.type == 'MESH':
            bpy.context.view_layer.objects.active = obj
            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.select_all(action='SELECT')
            bpy.ops.mesh.normals_make_consistent(inside=False)
            bpy.ops.object.mode_set(mode='OBJECT')

def enable_auto_smooth(self, context):
    bpy.ops.object.select_all(action='DESELECT')
    bpy.ops.object.select_by_type(type='MESH')
    
    for obj in bpy.context.selected_objects:
        if obj.data.use_auto_smooth == False:
            obj.data.use_auto_smooth = True

def duplicate_join_merge_by_distance(self, context):
    bpy.ops.object.select_all(action='DESELECT')
    bpy.ops.object.select_by_type(type='MESH')
    
    # Deselect active object
    bpy.context.selected_objects.remove(bpy.context.active_object)
    
    # Duplicate selected objects
    bpy.ops.object.duplicate(linked=False)
    
    # Join duplicated objects
    bpy.ops.object.join()
    
    # Enter Edit Mode and merge by distance
    bpy.ops.object.mode_set(mode='EDIT')
    bpy.ops.mesh.select_all(action='SELECT')
    bpy.ops.mesh.remove_doubles(threshold=0.001)  # You can adjust the threshold as needed
    bpy.ops.object.mode_set(mode='OBJECT')

def calculate_total_statistics():
    total_objects = len(bpy.context.selected_objects)
    total_vertices = sum(len(obj.data.vertices) for obj in bpy.context.selected_objects)
    total_edges = sum(len(obj.data.edges) for obj in bpy.context.selected_objects)
    total_triangles = sum(len(obj.data.polygons) for obj in bpy.context.selected_objects)
    return total_objects, total_vertices, total_edges, total_triangles

def propagate_shape_keys(self, context):
    for obj in bpy.context.selected_objects:
        if obj.type == 'MESH':
            # Make sure the object has shape keys
            if obj.data.shape_keys:
                bpy.context.view_layer.objects.active = obj
                bpy.ops.object.mode_set(mode='EDIT')
                bpy.ops.mesh.select_all(action='SELECT')
                
                # Iterate through all shape keys and set their value to 0
                for shape_key in obj.data.shape_keys.key_blocks:
                    if shape_key != obj.data.shape_keys.reference_key:
                        shape_key.value = 0.0  # Set the value to 0
                
                bpy.ops.object.mode_set(mode='OBJECT')

def blend_from_shape_keys(self, context):
    for obj in bpy.context.selected_objects:
        if obj.type == 'MESH':
            # Make sure the object has shape keys
            if obj.data.shape_keys:
                bpy.context.view_layer.objects.active = obj
                bpy.ops.object.mode_set(mode='EDIT')
                bpy.ops.mesh.select_all(action='SELECT')
                
                # Iterate through all shape keys and blend from them
                for shape_key in obj.data.shape_keys.key_blocks:
                    if shape_key != obj.data.shape_keys.reference_key:
                        shape_key.value = 0.0  # Set the value to 0 to blend from this shape key
                
                bpy.ops.object.mode_set(mode='OBJECT')

def set_normals_from_faces(self, context):
    for obj in bpy.context.selected_objects:
        if obj.type == 'MESH':
            bpy.context.view_layer.objects.active = obj
            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.select_all(action='SELECT')
            
            # Recalculate normals from faces
            bpy.ops.mesh.normals_make_consistent(inside=False)
            
            bpy.ops.object.mode_set(mode='OBJECT')

# Append functions from the provided script
def ClearAllObjVGroups(obj):
    for vertex_group in obj.vertex_groups:
        obj.vertex_groups.remove(vertex_group)

def TransferObjWeights(source, target):
    # Get global transformation matrices
    source_matrix = source.matrix_world.copy()
    target_matrix = target.matrix_world.copy()
    
    # Make target object active, toggle weight paint mode
    bpy.context.view_layer.objects.active = target
    bpy.ops.object.mode_set(mode='OBJECT')
    bpy.ops.paint.weight_paint_toggle()
    
    # Apply global transformations to the source and target objects
    source.location = mathutils.Vector((0, 0, 0))
    source.rotation_euler = mathutils.Euler((0, 0, 0))
    source.scale = mathutils.Vector((1, 1, 1))

    # Ensure that the target object has vertex groups
    if not target.vertex_groups:
        for source_group in source.vertex_groups:
            target.vertex_groups.new(name=source_group.name)

    # Perform 'Weight Transfer' operation with face interpolation
    bpy.ops.object.data_transfer(
        use_reverse_transfer=True,
        data_type='VGROUP_WEIGHTS',
        vert_mapping='POLYINTERP_NEAREST',
        use_object_transform=True,
        ray_radius=5,
        layers_select_src='NAME',
        layers_select_dst='ALL',
        mix_mode='REPLACE'
    )

    # Restore global transformations
    source.matrix_world = source_matrix
    target.matrix_world = target_matrix

def GetSelectedDestinationObjects():
    selected_objects = bpy.context.selected_objects
    # Filter out the source object and objects without vertex groups
    return [obj for obj in selected_objects if obj != bpy.context.active_object and obj.vertex_groups]

def FindArmatureModifier(obj):
    for modifier in obj.modifiers:
        if modifier.type == 'ARMATURE':
            return modifier.object
    return None

def RemoveArmatureModifiers(objects):
    for obj in objects:
        armature_modifiers = [modifier for modifier in obj.modifiers if modifier.type == 'ARMATURE']
        for modifier in armature_modifiers:
            obj.modifiers.remove(modifier)

def ParentToSameArmature(source, destination):
    source_armature = FindArmatureModifier(source)
    if source_armature:
        # Remove existing armature modifiers from destination objects
        RemoveArmatureModifiers(destination)
        
        for obj in destination:
            armature_modifier = obj.modifiers.new(name="Armature", type='ARMATURE')
            armature_modifier.object = source_armature

def heat_seek_rigging_no_vertex(source, destination_objects):
    for obj in bpy.context.scene.objects:
        if obj.type == 'MESH' and obj != source and not obj.vertex_groups:
            # Duplicate the source object
            duplicate_source = source.copy()
            duplicate_source.data = source.data.copy()
            bpy.context.collection.objects.link(duplicate_source)
            bpy.context.view_layer.update()

            # Link the duplicate source object to the collection of the destination object
            destination_object_collection = obj.users_collection[0]
            destination_object_collection.objects.link(duplicate_source)

            # Transfer weights from the duplicate source to the destination object
            ClearAllObjVGroups(obj)
            TransferObjWeights(duplicate_source, obj)

            # Parent the destination object to the same armature as the source
            ParentToSameArmature(source, [obj])

            # Remove the duplicate source object after the transfer
            bpy.data.objects.remove(duplicate_source)

            # Check if the destination object received weights
            if not obj.vertex_groups:
                print(f"Vertex group transfer failed for {obj.name}")
            else:
                print(f"Vertex group transfer completed for {obj.name}")

# Define the transfer_weights function
def transfer_weights(self, context):
    selected_objects = bpy.context.selected_objects

    # Ensure that at least two objects are selected
    if len(selected_objects) < 2:
        self.report({'ERROR'}, "Please select at least two objects.")
        return {'CANCELLED'}

    source_object = selected_objects[0]
    target_objects = selected_objects[1:]

    # Transfer weights from the source object to the target objects
    for target_object in target_objects:
        TransferObjWeights(source_object, target_object)

# Create a new tab in the N-panel
class VIEW3D_PT_2KStreamlineTools(bpy.types.Panel):
    bl_label = "2K Streamline Tools"
    bl_idname = "VIEW3D_PT_2KStreamlineTools"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "2K Streamline"

    def draw(self, context):
        layout = self.layout

        # Calculate total statistics
        total_objects, total_vertices, total_edges, total_triangles = calculate_total_statistics()

        # Display total statistics
        layout.label(text=f"Total Selected Objects: {total_objects}")
        layout.label(text=f"Total Vertices: {total_vertices}")
        layout.label(text=f"Total Edges: {total_edges}")
        layout.label(text=f"Total Triangles: {total_triangles}")

        layout.separator()  # Add a separator line

        # Create subpanels for categorization
        layout.label(text="Shading", icon='MATERIAL')
        subpanel = layout.column(align=True)
        
        # Add buttons for the Shading category
        subpanel.operator("mesh.recalculate_normals_outside")
        subpanel.operator("mesh.enable_auto_smooth")
        subpanel.operator("mesh.duplicate_join_merge_by_distance")

        layout.label(text="Shape Keys", icon='SHAPEKEY_DATA')
        subpanel = layout.column(align=True)
        
        # Add buttons for the Shape Keys category
        subpanel.operator("mesh.blend_from_shape_keys")
        subpanel.operator("mesh.propagate_shape_keys")

        layout.label(text="Auto Rigging", icon='POSE_HLT')
        subpanel = layout.column(align=True)
        
        # Add buttons for the Auto Rigging category
        subpanel.operator("mesh.transfer_weights")
        subpanel.operator("mesh.heat_seek_rigging_no_vertex")

# Register the operators and panel
def register():
    bpy.utils.register_class(VIEW3D_PT_2KStreamlineTools)
    bpy.utils.register_class(RecalculateNormalsOutsideOperator)
    bpy.utils.register_class(EnableAutoSmoothOperator)
    bpy.utils.register_class(DuplicateJoinMergeByDistanceOperator)
    bpy.utils.register_class(PropagateShapeKeysOperator)
    bpy.utils.register_class(BlendFromShapeKeysOperator)
    bpy.utils.register_class(SetNormalsFromFacesOperator)
    bpy.utils.register_class(TransferWeightsOperator)
    bpy.utils.register_class(HeatSeekRiggingNoVertexOperator)

def unregister():
    bpy.utils.unregister_class(VIEW3D_PT_2KStreamlineTools)
    bpy.utils.unregister_class(RecalculateNormalsOutsideOperator)
    bpy.utils.unregister_class(EnableAutoSmoothOperator)
    bpy.utils.unregister_class(DuplicateJoinMergeByDistanceOperator)
    bpy.utils.unregister_class(PropagateShapeKeysOperator)
    bpy.utils.unregister_class(BlendFromShapeKeysOperator)
    bpy.utils.unregister_class(SetNormalsFromFacesOperator)
    bpy.utils.unregister_class(TransferWeightsOperator)
    bpy.utils.unregister_class(HeatSeekRiggingNoVertexOperator)

if __name__ == "__main__":
    register()
